/*
*******************************************************************************
*
* File:         xmailbox.c
* Description:  
* Author:       Henrik Liebau
* Created:      19 May 2000, 11:00:32
* Modified:     Tue May 29 10:18:31 2001 by Henrik Liebau, BVS R&D
* Language:     C Source File
* Package:      
* Status:       
*
* (C) Copyright 2000 by Agilent Technologies, All Rights Reserved.
*
*******************************************************************************
*/

#ifdef WINNT
# include <xidentos.h>
# include <xpci32rg.h>
#endif  /* WINNT */

#include <xtypedef.h>
#include <xmailbox.h>
#include <xpci.h>

#ifndef MAILBOX_ONLY
#include <xregx10.h>

#include <xregcons.h> /* necessary for MAILBOX_ACCESS etc. */
#include <xerrcapi.h> /* necessary for prototype of BestXLastErrorParamSet..*/
#include <xsession.h> /* necessary for BestXStream2Long etc. */
#include <xiocommo.h> /* for BestXBasicCommand */
#endif /* ndef MAILBOX_ONLY */

/*---------------------------------------------------------------------------*
 * EXPORT bx_errtype BestXDevIdentifierGet(
 *
 * Purpose: search for device with vendor/device ID
 *---------------------------------------------------------------------------*/
bx_errtype EXPORT BestXDevIdentifierGet(
                                        bx_int32 vendor_id,
                                        bx_int32 device_id,
                                        bx_int32 number,
                                        bx_int32 * devid
                                        )
{
#if defined WINNT
  /* we can do this the easy way */
  return BestXNTRegDevIDGet(vendor_id, device_id, number, devid);
  
#elif defined _AIX
  /* This really isn't necessary from the AIX point of view as we 
   * do not require the bus Id as we get that from bx_setup_all_dma_buf
   * in bx_dd_interface.c
   */
  *devid = number;
  return BX_E_OK;

#else
  /* we're in DOS or other non-Win32 land */
  
  bx_int32 cNumFound = 0;
  bx_int32 slotnumber;
  bx_int32 value;
  bx_int32 LookForValue = (vendor_id | (device_id << 16));
  bx_errtype err;
  
  /* There are 256 busses, each consisting of 32 slots => 0x2000 slots */
  for (slotnumber = 0; slotnumber < 0x2000UL; slotnumber++)
  {
    /* check vendor & device id */
    err = BestXGenericPCIDWGet(slotnumber << 3, 0UL, &value);
    
    if (BX_E_OK == err)
    {
      if ((value == LookForValue) && (cNumFound++ == number))
      {
        /* we found the right BEST ... **SUCCESS** */
        *devid = slotnumber << 3;
        return BX_E_OK;
      }
    }
    else 
    {
      if (BX_E_NO_BEST_PCI_DEVICE_FOUND == err)
      {
        /* do nothing...this is an appropriate response */
      }
      else
      {
        /* a real error...return it */
        return err;
      }
    }
  }
  
  /* we didn't find the one the caller was looking for */
  *devid = 0xffffffffUL;
  return BX_E_NO_BEST_PCI_DEVICE_FOUND;
  
#endif /* defined WINNT */
}

/*---------------------------------------------------------------------------*
 * bx_errtype EXPORT BestXPCICfgMailboxReceiveRegRead(
 *
 * Purpose      : receive value from mailbox via PCI
 *---------------------------------------------------------------------------*/
bx_errtype EXPORT BestXPCICfgMailboxReceiveRegRead(
    bx_int32 devid,
    bx_int32 * value,
    bx_int32 * status
)
{
  bx_errtype err;
  err = BestXPCIMailboxRead(devid, value);

  *status = (err == BX_E_OK) ? 0x1 : 0;

  if (err)
  {
    *value = 0xffffffffUL;
  }

  /* BX_ERRETURN not possible */
  return BX_E_OK;
}


/*---------------------------------------------------------------------------*
 * bx_errtype EXPORT BestXPCICfgMailboxSendRegWrite(
 *
 * Purpose      : send value to mailbox via PCI
 *---------------------------------------------------------------------------*/
bx_errtype EXPORT BestXPCICfgMailboxSendRegWrite(
    bx_int32 devid,
    bx_int32 value,
    bx_int32 * status
)
{
  bx_errtype err = BestXPCIMailboxWrite(devid, value);
  *status = (err == BX_E_OK) ? 0x01 : 0;
  /* BX_ERRETURN not possible */
  return BX_E_OK;
}

/*****************************************************************************
 * the following functions are for mailbox access through a port other
 * than PCI
 *****************************************************************************/
#ifndef MAILBOX_ONLY

/*---------------------------------------------------------------------------*
 * bx_errtype EXPORT BestXMailboxReceiveRegRead(
 *
 * Purpose: 
 *---------------------------------------------------------------------------*/
bx_errtype EXPORT BestXMailboxReceiveRegRead(
    bx_handletype handle,
    bx_int32 * val,
    bx_int32 * status
)
{
  BX_TRY_VARS_NO_PROG;

  BX_TRY_BEGIN
  {
    bx_int8 data8;
    bx_int8 buf[OUT_MAILBOX_READ];
    bx_int8ptr bufptr;
    *val = 0xffffffffUL;        /* init for failure */
    *status = 0;
    
    BestXLastErrorParamSet(handle, BX_ERRPAR_1,(bx_int32)BX_PORT_COUNT + 1UL);
    BestXLastErrorParamSet(handle, BX_ERRPAR_2,(bx_int32)bx_handlearray[handle].port);
    BX_TRY_FAIL(((bx_handlearray[handle].port == BX_PORT_PCI_CONF) ||
                 (bx_handlearray[handle].port == BX_PORT_PCI_IO)) ?
                  BX_E_WRONG_PORT : BX_E_OK);
    
#ifdef CUSTOM_OEM1
    BX_TRY_FAIL((bx_handlearray[handle].port == BX_PORT_OEM) ? BX_E_WRONG_PORT : BX_E_OK);
#endif

    if(BestXHasFirmware(handle))
    {
      if (BestX16BitRegisterFile(handle))
      {
        bx_int16 outsize = OUT_MAILBOX_READ;

      
        BX_TRY(BestXBasicCommand(handle, CMD_MAILBOX_READ,
               NULL, IN_MAILBOX_READ, buf, &outsize));
      

        bufptr = BestXStream2Long(val, buf, 1UL);
        (void) BestXByteCopy(&data8, bufptr, 1UL);
        *status = data8;          /* msg not previously read */
      }
    }
    else
    {
      BX_TRY(BestXBasicBlockRead(handle, MAILBOX_ACCESS, &data8, 1UL));
      *status = 0x1;            /* msg not previously read */
      *val = (bx_int32) data8;
    }
  }
  
  BX_TRY_CATCH
  {
    *status = 0;
    
    if (BX_TRY_RET == BX_E_FUNC)
      BX_TRY_RET = BX_E_OK;       /* this is for 25 return mechanism */
  }
  
  BX_ERRETURN(BX_TRY_RET);
}


bx_errtype EXPORT BestXMailboxSendRegWrite(
    bx_handletype handle,
    bx_int32 val,
    bx_int32 * status
)
{
  BX_TRY_VARS_NO_PROG;
  bx_int8 data8;
  bx_int8 buf[sizeof(bx_int32)];
  BX_TRY_BEGIN
  {
    BestXLastErrorParamSet(handle, BX_ERRPAR_1, (bx_int32)BX_PORT_COUNT + 1UL);
    BestXLastErrorParamSet(handle,
                           BX_ERRPAR_2,
                           (bx_int32)bx_handlearray[handle].port
                           );
    BX_TRY_FAIL(((bx_handlearray[handle].port == BX_PORT_PCI_CONF) ||
        (bx_handlearray[handle].port == BX_PORT_PCI_IO)) ?
      BX_E_WRONG_PORT : BX_E_OK);

#ifdef CUSTOM_OEM1
    BX_TRY_FAIL((bx_handlearray[handle].port == BX_PORT_OEM) ? BX_E_WRONG_PORT : BX_E_OK);
#endif

    if(BestXHasFirmware(handle))
    {
      if (BestX16BitRegisterFile(handle))
      {
        bx_int16 outsize = OUT_MAILBOX_WRITE;
        (void) BestXLong2Stream(buf, &val, 1UL);

        BX_TRY(BestXBasicCommand(handle, CMD_MAILBOX_WRITE,
            buf, IN_MAILBOX_WRITE, &data8, &outsize));
      

        *status = data8;          /* msg not previously read */
    
      }
    }
  }

  BX_TRY_CATCH
  {
    *status = 0;

    if (BX_TRY_RET == BX_E_FUNC)
      BX_TRY_RET = BX_E_OK;       /* this is for 25 return mechanism */
  }

  BX_ERRETURN(BX_TRY_RET);
}

#

#endif /* not defined MAILBOX_ONLY */
